import { useEffect, useState } from 'react';
import { ButtonGroup, Col, Modal, Row, Table, Button } from 'react-bootstrap';
import { Marker, OverlayView } from '@react-google-maps/api';
import useEditablePolyline from '../../../../hooks/useEditablePolyline';
import tjBoxIcon from '../../../../assets/icons/tjBox.png';
import axiosInstance from '../../../../util/axiosInstance';
import { TjConnect } from './TjBoxConnect/TjConnect';
import { useDispatch } from 'react-redux';
import { deleteTjBox } from '../../../../store/asyncMethods/tjBoxMethod';

function TjBoxInfo({ tjBoxInfo, setShow }) {
    const dispatch = useDispatch();

    const onDeleteHandler = () => {
        if (!window.confirm('Are you sure you want to delete this TJ Box?'))
            return;

        dispatch(
            deleteTjBox(tjBoxInfo.id, () => {
                setShow(false);
            })
        );
    };

    return (
        <Col lg={6}>
            <Table striped bordered hover className='m-2'>
                <tbody>
                    <tr>
                        <td>ID</td>
                        <td>{tjBoxInfo.id}</td>
                    </tr>
                    <tr>
                        <td>Name</td>
                        <td>{tjBoxInfo.name}</td>
                    </tr>
                    <tr>
                        <td>Remarks</td>
                        <td>{tjBoxInfo.remarks}</td>
                    </tr>
                </tbody>
            </Table>
            <ButtonGroup className='mb-2 btn-group-sm'>
                <Button variant='warning'>update</Button>
                <Button variant='danger' onClick={onDeleteHandler}>
                    Delete
                </Button>
            </ButtonGroup>
        </Col>
    );
}

const TjBoxView = ({ tjBox, show, setShow }) => {
    const [tjBoxInfo, setTjBoxInfo] = useState(null);

    useEffect(() => {
        if (show) {
            const fetchTjBoxInfo = async () => {
                try {
                    const { data } = await axiosInstance.get(
                        `/tj-boxes/${tjBox.id}`
                    );
                    setTjBoxInfo(data);
                } catch (error) {
                    console.log(error);
                }
            };
            fetchTjBoxInfo();
        }
    }, [show, tjBox]);

    return (
        tjBoxInfo && (
            <Modal show={show} onHide={() => setShow(false)} fullscreen={true}>
                <Modal.Header closeButton>
                    <Modal.Title>
                        {tjBoxInfo.name} - Fiber Terminal Joint Box
                    </Modal.Title>
                </Modal.Header>
                <Modal.Body>
                    <Row>
                        <TjBoxInfo tjBoxInfo={tjBoxInfo} setShow={setShow} />
                        <Col lg={6} />
                        <TjConnect tjBoxInfo={tjBoxInfo} />
                    </Row>
                </Modal.Body>
            </Modal>
        )
    );
};

const TJBox = ({ tjBox }) => {
    const [showTjBoxPanel, setShowTjBoxPanel] = useState(false);
    const { setParent } = useEditablePolyline();
    const onDblClickHandler = (event) => {
        setParent(tjBox, event.latLng);
    };

    return (
        <>
            <Marker
                position={JSON.parse(tjBox.coordinates)}
                onDblClick={onDblClickHandler}
                icon={tjBoxIcon}
            />
            <OverlayView
                position={JSON.parse(tjBox.coordinates)}
                mapPaneName={OverlayView.OVERLAY_MOUSE_TARGET}
                getPixelPositionOffset={() => ({
                    x: +15,
                    y: -40,
                })}
            >
                <div
                    className='p-2 text-white bg-dark rounded pointer'
                    style={{ position: 'absolute', whiteSpace: 'nowrap' }}
                    onContextMenu={() => setShowTjBoxPanel(true)}
                >
                    <p className='m-0'>{tjBox.name}</p>
                </div>
            </OverlayView>
            <TjBoxView
                show={showTjBoxPanel}
                setShow={setShowTjBoxPanel}
                tjBox={tjBox}
            />
        </>
    );
};

export default TJBox;
