<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Pop;
use App\Models\User;
use App\Models\Client;
use App\Models\Income;
use App\Models\Balance;
use App\Models\Employee;
use App\Classes\AuthUser;
use App\Classes\MikrotikService\Mikrotik;
use App\Classes\BkashUser;
use App\Models\IncomeHead;
use App\Classes\MikrotikService\SyncWithMk;
use App\Models\Billpayment;
use App\Models\Clientsinfo;
use App\Models\BillGenerate;
use App\Models\BkashPayment;
use Illuminate\Http\Request;
use App\Classes\Notification;
use App\Services\ExpireCheck;
use App\Models\UserAccounting;
use App\Classes\SMS\PaymentSms;
use App\Models\CustomerAccount;
use App\Jobs\SendPaymentMailJob;
use App\Services\ClientServices;
use App\Models\BalanceAdjustment;
use App\Models\CompanyInformation;
use Illuminate\Support\Facades\DB;
use App\Classes\BillgenerateUpdate;
use App\Services\CommissionService;
use App\Jobs\ClientSyncJobForSingle;
use Brian2694\Toastr\Facades\Toastr;
use App\Services\GenerateMonthlyBill;
use App\Models\ReselleBalanceLogReport;
use Illuminate\Support\Facades\Session;
use App\Models\SubResellerBalanceLogReport;
use App\Http\Controllers\Clients\IdEnableDisableController;
use App\Http\Controllers\Reseller\ResellerRechargeController;
use App\Models\OtcPaymentLog;
use App\Models\Packages;
use App\Models\ResellerCommissionReference;
use App\Services\SecondMonthlyBillGenerateForDayToDay;

class PaymentController extends Controller
{
    //


    public function getPaymentHistory($id)
    {
        //    dd(auth()->user()->roles->pluck('name')->toArray());

        if (!auth()->user()->hasPermissionTo('payment-report')) {
            return redirect()->back()->with('error_message', 'Permission Error!');
        }

        $list = ClientServices::customerList('singleUser', $id);

        if (!isset($list->first()->bill_generate) || $list->first()->bill_generate == 'no') {
            Toastr::error('User History Not available', 'error');
            return redirect()->back();
        }


        return view('billing.payment', [
            'list'               => $list->get(), //Client::with('clientsinfo','packages','pop')->paginate(5),
            'page_title'         => 'View Payment History',
            'employees'          => Employee::all(),
            'incomeHead'         => IncomeHead::all(),
            'paymenthistory'     => Billpayment::with('user', 'income', 'employee')->whereClient_id($id)->orderBy('id', 'desc'),
            'bill_history'       => BillGenerate::whereClient_id($id)->orderBy('id', 'desc'),
            'balance_adjustment' => BalanceAdjustment::with('user')->where('client_id', $id)->orderBy('id', 'desc')->take(10)

        ]);
    }

    public function paymentLog($id)
    {
        return view('paymenthistory.paymentLog', [
            'paymenthistory'    => Billpayment::with('user', 'income', 'employee')->whereClient_id($id)->orderBy('id', 'desc'),
        ]);
    }

    public function clientAccount($id)
    {
        $r = CustomerAccount::where('client_id', $id)->first();

        $due = $r->dueAmount > 0 ? $r->dueAmount : 0;
        $currentbalance = $r->dueAmount < 0 ? abs($r->dueAmount) : 0;
        $otc_due = Clientsinfo::where('client_id', $id)->first()->otc_due ?? 0;
        $client = Client::find($id);


        if ($client->clients_status == 'active') {
            $status = 'Active';
            $class = 'bg-success';
        } elseif ($client->clients_status == 'deactive') {
            $status = 'Deactive';
            $class = 'bg-secondary';
        } elseif ($client->clients_status == 'expired') {
            $status = 'Expired';
            $class = 'bg-danger';
        } elseif ($client->clients_status == 'disable') {
            $status = 'Disabled';
            $class = 'bg-warning';
        }


        return response()->json([
            'totalPaidAmount'       => $r->totalPaid,
            'totalDueAmount'        => $due,
            'totalCurrentAmount'    => $currentbalance,
            'totalOtcAmount'        => $otc_due,
            'cStatus'               => $status,
            'listDue'               => $r->dueAmount,
            'cExpireDate'           => Carbon::parse($client->expire_date)->format('d-M-Y'),
            'class'                 => $class,
        ]);
    }




    public function payment($id)
    {
        // dd($id);

        if (!auth()->user()->hasPermissionTo('money-receipt-entry')) {
            return redirect()->route('clients.index')->with('error_message', 'Permission Error!');
        }

        $list = ClientServices::customerList('singleUser', $id);

        return view('billing.payment', [
            'list'              => $list->get(), //Client::with('clientsinfo','packages','pop')->paginate(5),
            'page_title'        => 'View Payment Information',
            'paymenthistory'    => Billpayment::with('user')->whereClient_id($id)->orderBy('id', 'desc'),
            'pagetype'          => 'payment',
            'incomeHead'        => IncomeHead::all(),
            'employees'         => Employee::all(),
            'bill_history'      => BillGenerate::whereClient_id($id)->orderBy('id', 'desc')->take(25)
        ]);
    }

    public function balanceAdjustment(Request $request)
    {
        // dd($request->all());

        // $client = Client::with('customerAccount')->find($request->client_id);
        $client = Client::with('customerAccount')->where('id', $request->client_id)->first();

        DB::beginTransaction();


        try {

            if (!$client->customerAccount->isDueAmountNegative || abs($client->customerAccount->dueAmount) < $request->amount || $request->amount <= 0) {
                DB::commit();
                Toastr::error('Sorry not permitted', 'error');
                return redirect()->back();
            }

            BalanceAdjustment::create([
                'amount'        => $request->amount,
                'description'   => $request->description,
                'user_id'       => auth()->user()->id,
                'client_id'     => $request->client_id
            ]);

            $customerAccounts = CustomerAccount::find($client->customerAccount->id);
            $customerAccounts->increment('dueAmount', $request->amount);

            DB::commit();
            Toastr::success('Balance Adjust Successfull', 'success');
            return redirect()->back();
        } catch (\Throwable $th) {

            DB::rollBack();
            Toastr::error('Something is wrong!!', 'error');
            return redirect()->back();
        }
    }



    public function checkLastPaymentTimeForThisUser($id)
    {
        $t = Billpayment::where('client_id', $id)
            ->latest('id')
            ->first();

        if ($t !== null) {
            $now = now();
            $last_entry = $t->created_at;

            if ($last_entry->diffInMinutes($now) < 2) {
                return 2 - $last_entry->diffInMinutes($now);
            }
        }
    }


    public function autoPayment($client_id, $bill_amount, $type, $incomeHeadId)
    {

        $receipt_id =  'AT' . today()->timestamp . round($bill_amount);

        // dd($incomeHeadId);

        $data = [
            'description'           =>  $type . ' Bill Collection',
            'paid_amount'           =>  $bill_amount,
            'discount_amount'       =>  0,
            'client_id'             =>  $client_id,
            'user_id'               =>  auth()->id(),
            'created_at'            =>  now(),
            'collected_by'          =>  0,
            'money_receipt_number'  =>  $this->generateUniqueId($client_id, $bill_amount),
            'payment_mthod'         =>  'cash',
            'client_id_time'        =>  $client_id . '_AT_' . now()->addMinute()->format('Y-m-d H:i')
        ];
        $payment = Billpayment::create($data);

        $income_type = IncomeHead::find($incomeHeadId);

        $income =   Income::create([
            'name' => $income_type->name,
            'date' => now(),
            'amount' => $bill_amount,
            'description' => 'Bill Payment for customer : ' . Client::find($client_id)->userid,
            'incomeHead' => $incomeHeadId
        ]);
        $payment->income_id = $income->id;
        $payment->save();

        $actions = 'Bill Payment. cost ' . $bill_amount . 'TK for the customer ' . $client_id;
        $UserAccounting = UserAccounting::userAcStore($bill_amount, $actions, $client_id);
        $payment->user_accountings_id = $UserAccounting->id;

        CustomerAccount::updateCustomrAccount($client_id, (-$bill_amount), $bill_amount, 0);
    }

    public function generateUniqueId($client, $bill_amount)
    {
        return 'AT' . now()->timestamp . round($bill_amount) . $client;
    }

    public function newUserPaymentProcess($id)
    {

        $billinfo = BillGenerate::where('client_id', $id)->first();


        if ($billinfo) {

            $client = Client::with('clientsinfo')->where('id', $id)->first();


            $data = [
                'description'           =>  'New User Bill Collection',
                'paid_amount'           =>  $billinfo->bill_amount,
                'discount_amount'       =>  0,
                'client_id'             =>  $id,
                'user_id'               =>  auth()->id(),
                'created_at'            =>  now(),
                'collected_by'          =>  0,
                'money_receipt_number'  =>  $this->generateUniqueId($client->id, $billinfo->bill_amount),
                'payment_mthod'         =>  'cash',
                'client_id_time'        =>  $id . '_' . now()->format('Y-m-d H:i')
            ];
            $payment = Billpayment::create($data);

            $income_type = IncomeHead::find(1);

            $income =   Income::create([
                'name' => $income_type->name,
                'date' => now(),
                'amount' => $billinfo->bill_amount,
                'description' => 'Bill Payment for customer : ' . $client->userid,
                'incomeHead' => 1
            ]);
            $payment->income_id = $income->id;
            $payment->save();


            CustomerAccount::updateCustomrAccount($id, (-$billinfo->bill_amount), $billinfo->bill_amount, 0);

            $cac = CustomerAccount::where('client_id', $id)->first();
        }
    }

    public function newUserOTCPaymentProcess($id, $otc, $income_id)
    {

        $client = Client::with('clientsinfo')->where('id', $id)->first();

        $receipt_id =  'AT' . today()->timestamp . round($otc);

        $data = [
            'description'           =>  'OTC Collection',
            'paid_amount'           =>  $otc,
            'discount_amount'       =>  0,
            'client_id'             =>  $id,
            'user_id'               =>  auth()->id(),
            'created_at'            =>  now(),
            'collected_by'          =>  0,
            'money_receipt_number'  =>  $this->generateUniqueId($client->id, $otc),
            'income_id'             => $income_id,
            'payment_mthod'         =>  'cash',
            'client_id_time'        =>  $id . '_OTC_' . now()->format('Y-m-d H:i')
        ];
        $payment = Billpayment::create($data);

        CustomerAccount::updateCustomrAccount($id, (-$otc), $otc, 0);
    }

    public function getClient($id)
    {
        $client = Client::with('clientsinfo', 'pop', 'pop.reseller', 'customerAccount', 'packages')
            ->with(['generatedBill' => function ($q) {
                $q->whereBetween('created_at', [today()->startOfDay(), today()->endOfDay()])->where('billing_type', '!=', 'deleted');
            }])
            ->withCount('generatedBill')->find($id);

        return $client;
    }

    private function generateMonthlyBillIfNotExist($client_id)
    {
        $bill = BillGenerate::where('client_id', $client_id)
            ->whereBetween('created_at', [today()->startOfDay(), today()->endOfDay()])
            ->where('billing_type', 'monthly')
            ->first();

        if (!$bill) {
            $user = Client::find($client_id);
            (new GenerateMonthlyBill)->generate($user);

            if (getBillingType() === 'day_to_day' && $user->expire_date < today()) {
                $user->expire_date = today();
                $user->billing_cycle = date('d');
                $user->save();
            }
        }
    }

    private function single_bill_amount_for_day_to_day($client, $payment_day)
    {
        $amount = $client->packages->package_rate - $client->parmanent_discount;
        return round(($amount / 30) * $payment_day);
    }


    public function billGenerateIfNotExist($client_id, $payment_day = null)
    {
        $client = Client::find($client_id);
        if (checkSettings('auto-bill-generate-on-payment') == 'enable') {

            if ($payment_day && $payment_day == 30) {
                $this->generateMonthlyBillIfNotExist($client_id);
            } elseif ($payment_day) {

                $client = Client::with('packages', 'clientsinfo')->find($client_id);
                $bill_amount = $this->single_bill_amount_for_day_to_day($client, $payment_day);
                $note = 'Bill payment for ' . $payment_day . ' days';
                $bill_type = 'day_to_day';
                CustomerAccount::updateCustomrAccount($client_id, $bill_amount, 0, 0, $bill_amount);
                BillGenerateController::entryBillGenerate($client_id, $bill_amount, $note, $bill_type, today());
            } else {
                $this->generateMonthlyBillIfNotExist($client_id);
            }
        }
    }





    public function paymentprocess(Request $request)
    {

        if (!auth()->user()->hasPermissionTo('money-receipt-entry')) {
            return response()->json([
                'status' => 'error',
                'message' => 'Permission Error!!'
            ]);
        }
        if ($request->collected_by == null) {
            return response()->json([
                'status' => 'error',
                'message' => 'Please select Collected By!'
            ]);
        }

        if ($request->discount != null && $request->amount == null) {
            return response()->json([
                'status' => 'error',
                'message' => 'Amount must be 0 or greater than 0'
            ]);
        }

        $this->billGenerateIfNotExist($request->id, $request->payment_day);

        $client = $this->getClient($request->id);

        if (getBillingType() === 'day_to_day' && $request->amount + $request->discount < 1) {

            if ($request->payment_day) {
                $amount = round(($client->packages->package_rate / 30) * $request->payment_day);
            } else {
                $amount = $client->packages->package_rate;
            }
            $request->request->add(['amount' => $amount]);
        } elseif ($request->amount + $request->discount < 1) {

            return response()->json([
                'status' => 'error',
                'message' => 'Amount must be greater than 0'
            ]);
        }

        if (!$request->payment_day) {
            $request->merge(['payment_day' => 30]);
        }

        if ($request->amount < 1 && $request->discount > 0 && $request->payment_type  == 2) {
            return response()->json([
                'status' => 'error',
                'message' => 'OTC must be greter then 0'
            ]);
        }


        if (!empty($request->money_receipt_number)) {
            if (Billpayment::where('money_receipt_number', $request->money_receipt_number)->first() != null) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Duplicate MR ID'
                ]);
            }
            $receipt_id = $request->money_receipt_number;
        } else {
            if (checkSettings('money_receipt_no_required') == 'enable') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Money Receipt No is required'
                ]);
            } else {
                $receipt_id =  $this->generateUniqueId($client->id, $request->bill_amount);
            }
        }


        $current_due_amount = $client->customerAccount->dueAmount ?? 0;


        if ($request->payment_type == 2 && $client->clientsinfo->otc_due < $request->amount) {
            return response()->json([
                'status' => 'error',
                'message' => 'OTC must be same as Due OTC'
            ]);
        }

        if ($this->checkLastPaymentTimeForThisUser($client->id) > 0) {
            return response()->json([
                'status' => 'error',
                'message' => 'Please Try After ' . $this->checkLastPaymentTimeForThisUser($client->id) . ' Minutes.'
            ]);
        }

        DB::beginTransaction();

        try {


            $due_adjust_amount = $request->amount ?? 0 + $request->discount ?? 0;

            $month = $request->month;


            $data = [
                'description'           =>  $request->description ?? ' Bill Collection',
                'paid_amount'           =>  $request->amount ?? 0,
                'discount_amount'       =>  $request->discount ?? 0,
                'client_id'             =>  $request->client_id,
                'user_id'               =>  auth()->user()->id,
                'created_at'            =>  now(),
                'collected_by'          =>  $request->collected_by,
                'money_receipt_number'  =>  $receipt_id,
                'expire_date_history'   =>  $client->expire_date,
                'payment_method'        =>  'cash',
                'client_id_time'        =>  $request->client_id . '_' . now()->format('Y-m-d H:i')

            ];
            $payment = Billpayment::create($data);

            CustomerAccount::updateCustomrAccount(
                $request->client_id,
                (-$request->amount),
                $request->amount,
                $request->discount ?? 0
            );

            if ($client->pop->reseller->reseller_type == 'own') {
                $actions = 'Bill Payment. cost ' . $due_adjust_amount . 'TK for the customer ' . $client->userid;
                $UserAccounting = UserAccounting::userAcStore($request->amount, $actions, $client->id);
                $payment->user_accountings_id = $UserAccounting->id;
            }

            $income_type = IncomeHead::find($request->payment_type);
            $income =   Income::create([
                'name' => $income_type->name,
                'date' => now(),
                'amount' => $request->amount,
                'description' => 'Bill Payment for customer : ' . $client->userid,
                'incomeHead' => $request->payment_type
            ]);
            $payment->income_id = $income->id;
            $payment->save();


            if ($request->payment_type == 2) { //type 2 meening new customer
                $clientinfo = clientsinfo::where('client_id', $client->id)->first();
                $clientinfo->otc_due = $clientinfo->otc_due - ($request->amount + $request->discount);
                $clientinfo->save();
            }


            $cac = CustomerAccount::where('client_id', $client->id)->first();
            $st = $this->performeExpireDateExtend($client, $payment, $cac, $request->amount, $client->pop, $current_due_amount, '', $request->discount, $request->payment_day);

            if ($st == 'balance_error') {
                $pop = Pop::where('id', $client->pop_id)->first();

                if ($pop->subreseller == 'yes') {
                    return response()->json([
                        'status' => 'error',
                        'message' => ' এর ব্যালেন্স নাই'
                    ]);
                } else {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'ম্যানেজার এর ব্যালেন্স নাই'
                    ]);
                }
            }

            DB::commit();

            try {

                (new BillgenerateUpdate)->update($client->id);
            } catch (\Exception $e) {
            }

            try {

                if ($request->sms == 'true') {

                    if ($client->pop->sms_send == 'yes') {

                        $cac = CustomerAccount::where('client_id', $client->id)->first();
                        $totalDue = $cac->dueAmount < 0 ? 0 : $cac->dueAmount;
                        (new PaymentSms)->billPaymentSms($this->getClient($request->id), $request->amount, $receipt_id, $payment->id, 'Cash', $request->discount ?? 0, $totalDue ?? 0, $request->description, $request->payment_type);
                    }
                }
            } catch (\Exception $e) {
            }

            try {

                $emailSetting = json_decode(emailSetting()->where('tamplate_name', 'customer_payment_eamil')->first()->email_body);
                if ($emailSetting->sendemail == 'Yes') {
                    SendPaymentMailJob::dispatch($client, $payment, $emailSetting);
                }
            } catch (\Exception $e) {
            }

            try {
                (new Notification)->notify("..::[Bill Payment]::..\n Payment By: " .
                    auth()->user()->email . " \nUsername : " .
                    $client->userid . "\nAmount: " . $request->amount . "\nReceipt ID: " . $receipt_id . "\nDescription: " . $request->description, $client->pop->reseller->reseller_type);
            } catch (\Exception $e) {
            }

            try {

                if (checkAPI()) {
                    ClientSyncJobForSingle::dispatch($client->id);
                }
            } catch (\Exception $e) {
            }


            return response()->json([
                'status' => 'success',
                'message' => 'Payment Done!!',
                'payment_id' => $payment->id
            ]);
        } catch (\Throwable $th) {
            DB::rollback();
            return response()->json([
                'status' => 'error',
                'message' => 'Something Is wrong Please check on customer payment ' . $th
            ]);
        }
    }




    public function billPayPayment($trxid, $client_id, $amount, $mobile, $datetime, $type, $billpay = '')
    {
        if ($trxid && $client_id && $amount && $mobile) {

            $this->billGenerateIfNotExist($client_id);


            $payment = Billpayment::where('money_receipt_number', $type . '_' . $trxid)->first();

            if ($payment == null) {

                DB::beginTransaction();
                try {

                    $client = Client::with('clientsinfo', 'pop', 'pop.reseller:id,billable,reseller_type', 'customerAccount')
                        ->withCount('generatedBill')
                        ->where('id', $client_id)
                        ->first();

                    $current_due_amount = $client->customerAccount->dueAmount;


                    $user = (new AuthUser)->check('user@' . $client->app_key_type . '.com', $client->app_key_type);


                    $data = [
                        'description'           =>  'Bill Payment from bill pay (' . $client->app_key_type . ')',
                        'paid_amount'           =>  $amount ?? 0,
                        'discount_amount'       =>  0,
                        'client_id'             =>  $client->id,
                        'user_id'               =>  $user->id,
                        'created_at'            =>  now(),
                        'collected_by'          =>  Employee::onlinePaymentEmployee($client->app_key_type)->id,
                        'payment_method'        =>  'bill_pay',
                        'money_receipt_number'  =>  $client->app_key_type . '_' . $trxid,
                        'expire_date_history'   =>  $client->expire_date,
                        'trx_id'                =>  $trxid,
                        'customerMsisdn'        =>  $mobile,
                        'paymentCreateTime'     =>  Carbon::parse($datetime),
                        'client_id_time'        =>  $client->id . '_' . now()->format('Y-m-d H:i')
                    ];

                    $payment = Billpayment::create($data);

                    CustomerAccount::updateCustomrAccount($client->id, (-$amount), $amount, 0);


                    if ($client->pop->reseller->reseller_type == 'own') {

                        $actions = 'Bill Payment. cost ' . $amount . 'TK for the customer ' . $client->userid;

                        $userAccountingId = UserAccounting::create([
                            'received_amount'   => $amount,
                            'paid_amount'       => 0,
                            'actions'           => $actions,
                            'commetns'          => '',
                            'user_id'           => $user->id,
                            'client_id'         => $client->id,
                            'accounting_type'   => $client->app_key_type . ' Bill Pay',
                            'transaction_type'  => $client->app_key_type . ' Bill Pay',
                        ]);
                        $payment->user_accountings_id = $userAccountingId;
                    }


                    $income_type = IncomeHead::find(12);

                    $income =   Income::create([
                        'name' => $income_type->name ?? 'Bill Payment',
                        'date' => now(),
                        'amount' => $amount,
                        'description' => 'Bill Payment from customer : ' . $client->userid,
                        'incomeHead' => 11
                    ]);

                    $payment->income_id = $income->id;
                    $payment->save();

                    $cac = CustomerAccount::where('client_id', $client->id)->first();
                    // dd($amount);
                    $this->performeExpireDateExtend($client, $payment, $cac, $amount, $client->pop, $current_due_amount, $billpay);

                    $client->save();

                    DB::commit();

                    try {
                        if (checkSettings('auto-payment-sms-send') == 'enable') {
                            (new PaymentSms)->billPaymentSms($client, $amount, $trxid, $payment->id);
                        }
                    } catch (\Throwable $e) {
                    }

                    try {
                        if (checkAPI()) {
                            ClientSyncJobForSingle::dispatch($client->id);
                        }
                    } catch (\Throwable $e) {
                    }


                    // return 'success';
                    return ['status' => 'success', 'message' => 'success'];
                } catch (\Throwable $th) {

                    DB::rollBack();
                    return ['status' => 'duplicate_error', 'message' => $th->getMessage()];
                    // return 'duplicate_error';
                }
            } else {
                return ['status' => 'trx_id_error', 'message' => 'trx_id_error'];
                // return 'trx_id_error';
            }
        }
    }

    public function bkashPaymentProcess($bkash_paymentID, $api)
    {
        // dd("comes");

        if ($api != 'bkashPaymentFromClient') {
            Toastr::error('Something is worng', 'error');
            return redirect()->back();
        }

        $bkash = BkashPayment::where('paymentID', $bkash_paymentID)->where('transactionStatus', 'Completed')->first();

        $this->billGenerateIfNotExist($bkash->client_id);

        $payment = Billpayment::where('money_receipt_number', $bkash->paymentID)->first();

        if ($payment != null) {
            return 'success';
        }

        $client = Client::with('clientsinfo', 'pop', 'pop.reseller', 'customerAccount', 'generatedBill')->withCount('generatedBill')->find($bkash->client_id);


        $current_due_amount = $client->customerAccount->dueAmount ?? 0;


        $bill_payment_check = Billpayment::where('money_receipt_number', $bkash->trxID)->first();

        if (!empty($bkash->amount) && $bkash->transactionStatus == 'Completed' && $bill_payment_check == null) {

            $bkashUser = (new BkashUser)->check();

            $bkashUser = (new AuthUser)->check('user@bkash.com', 'Bkash Online Payment');

            $due_adjust_amount = $bkash->amount;

            $month = now()->month();

            $data = [
                'description'           =>  'Bkash Bill Payment from client portal',
                'paid_amount'           =>  $bkash->amount ?? 0,
                'discount_amount'       =>  0,
                'client_id'             =>  $bkash->client_id,
                'user_id'               =>  User::onlinePaymentUser('Bkash-Online')->id,
                'created_at'            =>  now(),
                'collected_by'          =>  Employee::onlinePaymentEmployee('Bkash-Online')->id,
                'money_receipt_number'  =>  $bkash->trxID,
                'expire_date_history'   =>  $client->expire_date,
                'payment_method'        =>  'bkash',
                'client_id_time'        =>  $bkash->client_id . '_' . now()->format('Y-m-d H:i')
            ];

            $payment = Billpayment::create($data);



            // dd($due_adjust_amount);


            CustomerAccount::updateCustomrAccount($bkash->client_id, (-$bkash->amount), $bkash->amount, 0);

            if ($client->pop->reseller->reseller_type == 'own') {

                $actions = 'Bill Payment. cost ' . $bkash->amount . 'TK for the customer ' . $client->userid;

                $userAccountingId = UserAccounting::create([
                    'received_amount'   => $bkash->amount,
                    'paid_amount'       => 0,
                    'actions'           => $actions,
                    'commetns'          => '',
                    'user_id'           => $bkashUser->id,
                    'client_id'         => $bkash->client_id,
                    'accounting_type'   => 'Bkash Payment',
                    'transaction_type'  => 'Bkash Payment',
                ]);
                $payment->user_accountings_id = $userAccountingId;
            }

            $income_type = IncomeHead::find(11);

            $income =   Income::create([
                'name' => $income_type->name ?? 'Bkash Payment',
                'date' => now(),
                'amount' => $bkash->amount,
                'description' => 'Bkash Bill Payment from customer : ' . $client->userid,
                'incomeHead' => 11
            ]);
            $payment->income_id = $income->id;
            $payment->save();


            $cac = CustomerAccount::where('client_id', $client->id)->first();

            $this->performeExpireDateExtend($client, $payment, $cac, $bkash->amount, $client->pop, $current_due_amount, 'billpay', 0);

            if (checkSettings('auto-payment-sms-send') == 'enable') {
                try {

                    (new PaymentSms)->billPaymentSms($this->getClient($client->id), $bkash->amount, $bkash->trxID, $payment->id, 'bKash');
                } catch (\Exception $e) {
                }
            }
            if (checkAPI()) {
                ClientSyncJobForSingle::dispatch($client->id)->delay(now()->addMinutes(5));
                // ClientSyncJobForSingle::dispatch($client->id)->delay(now()->addMinutes(5));
            }
            Session::flash('success_message', 'Payment is Successful');
        }
    }


    public function performeExpireDateExtend($client, $payment, $cac, $due_adjust_amount, $pop, $current_due_amount, $billpay = '', $discount = 0, $total_day = 30)
    {
        // dd($billpay);
        $auto_recharge =  (checkSettings('online_payment_auto_recharge') == 'enable' && $billpay) ? True : False;
        if (getBillingType() == 'day_to_day') {
            $auto_recharge = $billpay ? True : False;
            if (checkSettings('check_manager_balance') == 'enable' && $billpay == '') {
                $billpay = "";
            } else {
                $billpay = 'day_to_day';
            }
        }

        $all_pay_bill_auto_payment_to_reseller_sub_reseller = (checkSettings('all_pay_bill_auto_recharge') == 'enable' && $billpay) ? True : False;

        $income = Income::find($payment->income_id);

        $reseller_deducted_amount       = ReselleBalanceLogReport::where('client_id', $client->id)->sum('amount');
        $sub_reseller_deducted_amount   = SubResellerBalanceLogReport::where('client_id', $client->id)->sum('amount');

        $total_customer_paid = ($cac->totalPaid);

        $reseller_deductable_amount     = $total_customer_paid - $reseller_deducted_amount;
        $sub_reseller_deductable_amount = $total_customer_paid - $sub_reseller_deducted_amount;


        if ($reseller_deductable_amount > 0) {

            $curResellerBalance = ExpireCheck::getResellerBalance('reseller', $pop->reseller_id);

            if ($reseller_deductable_amount > $curResellerBalance && $billpay == '') {
                return 'balance_error';
            }

            if ($auto_recharge) {
                (new ResellerRechargeController)->rechargeByOnlinePayment($pop->reseller_id, $due_adjust_amount, $client, $billpay);
            }

            $this->reseller_balance_deduct($pop, $reseller_deductable_amount, $client, $payment);
        }

        if ($sub_reseller_deductable_amount > 0) {

            if ($pop->subreseller == 'yes') {

                $cur_sub_reseller_balance = ExpireCheck::getResellerBalance('pop', $pop->id);

                if ($sub_reseller_deductable_amount > $cur_sub_reseller_balance && $billpay == '') {
                    return 'balance_error';
                }

                if ($auto_recharge) {
                    $remark = 'Recharge By Online Payment. Customer ' . $client->userid . ' CID: ' . $client->id;
                    (new PopController)->rechargeByOnlinePayment('pop', $pop->id, $due_adjust_amount, $remark, $due_adjust_amount, 1);
                }

                $this->sub_reseller_balance_deduct($pop, $sub_reseller_deductable_amount, $client, $payment);
            }
        }

        if ($all_pay_bill_auto_payment_to_reseller_sub_reseller) {

            if ($pop->subreseller == 'yes') {
                $remark = 'Recharge By Online Payment. Customer ' . $client->userid . ' CID: ' . $client->id;
                (new ResellerRechargeController)->rechargeByOnlinePayment($pop->reseller_id, $payment->paid_amount, $client, $billpay);
                (new PopController)->rechargeByOnlinePayment('pop', $pop->id, $payment->paid_amount, $remark, $payment->paid_amount, 1);
            } else {

                (new ResellerRechargeController)->rechargeByOnlinePayment($pop->reseller_id, $payment->paid_amount, $client, $billpay);
            }
        }



        $exp = Carbon::parse($client->expire_date);

        $month_dif = 0;
        $dif = $exp->diffInMonths(today());
        $cac = CustomerAccount::where('client_id', $client->id)->first();


        if ($cac->dueAmount <= 0 && ($cac->dueAmount - abs($client->clientsinfo->otc_due)) <= 0) {
            $this->onPaymentUpdateInfo($client, $exp, $month_dif, $payment->id, $total_day);
        }
    }


    public function uddoktapay($request, $UddoktaPayLog = null)
    {

        $gateway = 'Uddokta-Pay';

        $info = array(
            'customer_id'    => $request['metadata']['customer_id'],
            'amount'         => $request['amount'],
            'transaction_id' => $request['transaction_id'],
            'description'    => 'Online Bill Payment By: ' . $gateway . ' Method: ' . $request['payment_method'],
            'payment_method' => $request['payment_method'],
        );

        if ($info['customer_id'] == null) {
            return response()->json(['status' => 'error', 'message' => 'Customer ID not found']);
        }

        // $client = ClientServices::customerinfo($info['customer_id'])->first();

        $this->billGenerateIfNotExist($request['metadata']['customer_id']);

        $client = Client::with('clientsinfo', 'pop', 'pop.reseller', 'customerAccount', 'generatedBill')->withCount('generatedBill')->find($info['customer_id']);
        if (!$client) {
            return response()->json(['status' => 'error', 'message' => 'Customer not found']);
        }

        //check transaction in bill payment if already exist return error response
        $billPayment = Billpayment::where('money_receipt_number', 'UD_' . $info['transaction_id'])->first();
        if ($billPayment) {
            return response()->json(['status' => 'error', 'message' => 'Transaction already exist']);
        }



        $payment = Billpayment::create([
            'description'           =>  $info['description'],
            'paid_amount'           =>  $info['amount'] ?? 0,
            'discount_amount'       =>  0,
            'client_id'             =>  $info['customer_id'],
            'user_id'               =>  User::onlinePaymentUser($gateway)->id,
            'created_at'            =>  now(),
            'collected_by'          =>  Employee::onlinePaymentEmployee($gateway)->id,
            'money_receipt_number'  =>  'UP_' . $info['transaction_id'],
            'trx_id'                =>  $info['transaction_id'],
            'payment_method'        =>  'Online Payment ' . $info['payment_method'],
            'client_id_time'        =>  $info['customer_id'] . '_' . now()->format('Y-m-d H:i')
        ]);

        $income =   Income::create([
            'name' => 'Online Payment by ' . $gateway,
            'date' => now(),
            'amount' => $info['amount'] ?? 0,
            'description' => 'Online Bill Payment (' . $gateway . ')  for customer : ' . $client->userid,
            'incomeHead' => 11
        ]);
        $payment->income_id = $income->id;
        $payment->save();

        CustomerAccount::updateCustomrAccount($info['customer_id'], (-$info['amount']), $info['amount'], 0);
        $cac = CustomerAccount::where('client_id', $client->id)->first();



        $current_due_amount = $client->customerAccount->dueAmount ?? 0;


        $status =  $this->performeExpireDateExtend($client, $payment, $cac, $info['amount'], $client->pop, $current_due_amount, 'billpay', 0);

        if (checkAPI()) {
            ClientSyncJobForSingle::dispatch($client->id);
        }


        if ($status == 'balance_error') {
            return 'error';
        } else {
            if (checkSettings('auto-payment-sms-send') == 'enable') {
                (new PaymentSms)->billPaymentSms($client, $info['amount'], $info['transaction_id'], $payment->id, 'uddoktapay');
            }

            // check if payment email will be send
            $emailSetting = json_decode(emailSetting()->where('tamplate_name', 'customer_payment_eamil')->first()->email_body);
            if ($emailSetting->sendemail == 'Yes') {
                SendPaymentMailJob::dispatch($client, $payment, $emailSetting);
            }

            $info = array(
                'id' => $request['metadata']['customer_id'],
                'login_status' => 'success'
            );

            Session::put('customer_info', json_encode($info));
            // return response()->json(['status' => 'success', 'message' => 'Payment Completed Successfully']);

            //return success response message

            if ($UddoktaPayLog) {
                $UddoktaPayLog->status = 'done';
                $UddoktaPayLog->save();
            }
            return response()->json(['status' => 'success', 'message' => 'Payment Completed Successfully']);
        }



        // return redirect()->route('customerDashboard')->with('success_message', 'Payment Completed Successfully');
    }



    public function sslpayment(Request $request)
    {
        if ($request->gateway && $request->gateway == 'uddoktapay') {
            $gateway = 'Uddokta-Pay';
        } else {
            $gateway = 'SSL-Commerz';
        }

        $info = array(
            'customer_id'    => $request->value_a,
            'amount'         => $request->amount,
            'transaction_id' => $request->tran_id,
            'description'    => 'Online Bill Payment: ' . $gateway
        );

        $this->billGenerateIfNotExist($request->value_a);


        // $client = ClientServices::customerinfo($info['customer_id'])->first();

        $client = Client::with('clientsinfo', 'pop', 'pop.reseller', 'customerAccount', 'generatedBill')->withCount('generatedBill')->find($info['customer_id']);


        $payment = Billpayment::create([
            'description'           =>  $info['description'],
            'paid_amount'           =>  $info['amount'] ?? 0,
            'discount_amount'       =>  0,
            'client_id'             =>  $info['customer_id'],
            'user_id'               =>  User::onlinePaymentUser($gateway)->id,
            'created_at'            =>  now(),
            'collected_by'          =>  Employee::onlinePaymentEmployee($gateway)->id,
            'money_receipt_number'  =>  'SSL_' . $info['transaction_id'],
            'trx_id'                =>  $info['transaction_id'],
            'payment_method'        =>  'Online Payment',
            'client_id_time'        =>  $info['customer_id'] . '_' . now()->format('Y-m-d H:i')
        ]);

        $income =   Income::create([
            'name' => 'Online Payment by ' . $gateway,
            'date' => now(),
            'amount' => $info['amount'] ?? 0,
            'description' => 'Online Bill Payment (' . $gateway . ')  for customer : ' . $client->userid,
            'incomeHead' => 11
        ]);
        $payment->income_id = $income->id;
        $payment->save();

        CustomerAccount::updateCustomrAccount($info['customer_id'], (-$info['amount']), $info['amount'], 0);
        $cac = CustomerAccount::where('client_id', $client->id)->first();



        $current_due_amount = $client->customerAccount->dueAmount ?? 0;


        // $this->performeExpireDateExtendForOnlinePayment($client, $info, $payment, $cac);

        $status =  $this->performeExpireDateExtend($client, $payment, $cac, $info['amount'], $client->pop, $current_due_amount, 'billpay', 0);

        if (checkAPI()) {
            ClientSyncJobForSingle::dispatch($client->id);
        }


        if ($status == 'balance_error') {
            return 'error';
        } else {
            if (checkSettings('auto-payment-sms-send') == 'enable') {
                try {
                    (new PaymentSms)->billPaymentSms($client, $info['amount'], $info['transaction_id'], $payment->id, 'sslpayment');
                } catch (\Exception $e) {
                }
            }

            // check if payment email will be send
            $emailSetting = json_decode(emailSetting()->where('tamplate_name', 'customer_payment_eamil')->first()->email_body);
            if ($emailSetting->sendemail == 'Yes') {
                try {
                    SendPaymentMailJob::dispatch($client, $payment, $emailSetting);
                } catch (\Exception $e) {
                }
            }

            $info = array(
                'id' => $request->value_a,
                'login_status' => 'success'
            );

            $request->session()->put('customer_info', json_encode($info));

            return 'success';
        }



        // return redirect()->route('customerDashboard')->with('success_message', 'Payment Completed Successfully');
    }


    public function performeExpireDateExtendForOnlinePayment($client, $info, $payment, $cac)
    {
        $pop = Pop::find($client->pop_id);
        $reseller_deducted_amount = ReselleBalanceLogReport::where('client_id', $client->id)->sum('amount');
        $sub_reseller_deducted_amount = SubResellerBalanceLogReport::where('client_id', $client->id)->sum('amount');

        $total_customer_paid = ($cac->totalPaid + $cac->totalDiscount);

        $reseller_deductable_amount = round($total_customer_paid - $reseller_deducted_amount, 2);
        $sub_reseller_deductable_amount = round($total_customer_paid - $sub_reseller_deducted_amount, 2);


        $this->reseller_balance_deduct($pop, $reseller_deductable_amount, $client, $payment);


        if ($pop->subreseller == 'yes') {
            $this->sub_reseller_balance_deduct($pop, $sub_reseller_deductable_amount, $client, $payment);
        }

        if ($client->clients_status == 'expired' && $client->expire_date <= now()) {

            if ($cac->dueAmount < 0 && abs($cac->dueAmount) >= $client->package_rate) {

                $exp = Carbon::parse($client->expire_date);
                if ($exp->diffInMonths(now()) < 1) {

                    $this->onPaymentUpdateInfo($client, $exp, 1, $payment->id);
                } else {
                    $exp = Carbon::parse(date('Y-m-' . $client->billing_cycle));
                    $this->onPaymentUpdateInfo($client, $exp, 1, $payment->id);
                }
            }
        }
    }


    public function reseller_balance_deduct($pop, $reseller_deductable_amount, $client, $payment)
    {
        // dd($client);
        if (isset(auth()->user()->email)) {
            $created_by = auth()->user()->email;
        } else {
            $created_by = 'Online Payment';
        }


        Balance::balanceDeductForUserRecharge('reseller', $pop->reseller_id, $reseller_deductable_amount);

        if ($client->packages->commission > 0) {
            // dd($payment);

            (new CommissionService())->reseller_commission($reseller_deductable_amount, 'payment', $pop->reseller_id, $client->packages->commission, $client->packages->package_rate, $client->id, "payment", $payment->id);
        }

        $report = ReselleBalanceLogReport::create(
            [
                'reseller_id' => $pop->reseller_id,
                'client_id'   => $client->id,
                'action'      => 'Bill payment from payment option payment id: ' . $payment->id,
                'amount'      => $reseller_deductable_amount,
                'remarks'     => 'Created BY ' . $created_by
            ]
        );

        $payment->reseller_balance_deduct = $reseller_deductable_amount;
        $payment->reseller_balance_log_report = $report->id;
        $payment->save();
    }


    public function sub_reseller_balance_deduct($pop, $sub_reseller_deducted_amount, $client, $payment)
    {
        $created_by = auth()->user() ? auth()->user()->email : 'Online Payment';
        Balance::balanceDeductForUserRecharge('pop', $pop->id, $sub_reseller_deducted_amount);

        $report = SubResellerBalanceLogReport::create(
            [
                'reseller_id'       => $pop->reseller_id,
                'client_id'         => $client->id,
                'action'            => 'Bill payment from payment option payment id: ' . $payment->payment_id,
                'amount'            => $sub_reseller_deducted_amount,
                'sub_reseller_id'   => $pop->id,
                'remarks'           => 'Created BY ' . $created_by
            ]
        );

        $payment->pop_balance_deduct = $sub_reseller_deducted_amount;
        $payment->sub_reseller_balance_log_report = $report->id;
        $payment->save();
    }

    public function resellerBalanceDeductCheck($pop, $extra_balance, $client, $cac, $payment)
    {
        if ($pop->billable == 'yes') {

            $curResellerBalance = ExpireCheck::getResellerBalance('reseller', $pop->reseller_id);

            if ($extra_balance > $curResellerBalance) {
                return redirect()->back()->with('error_message', 'Not Enough Balance');
            }

            Balance::deductBalanceOnPaymentFromReseller($client->id, $pop->reseller_id, $extra_balance, $cac, $payment->id);
        }
    }




    public function onPaymentUpdateInfo($client, $expdate, $month, $payment_id, $total_day = 30)
    {

        $current_month_billing_date = Carbon::parse(date($client->billing_cycle . '-M-Y'));

        $bill = BillGenerate::where('client_id', $client->id)
            ->whereBetween('created_at', [today()->firstOfMonth(), today()->endOfMonth()])
            ->where('billing_type', 'monthly')
            ->first();

        $date = $client->expire_date;


        if ($bill) {
            if (Carbon::parse($client->expire_date) < today()->endOfMonth()) {

                $date = Carbon::parse($current_month_billing_date)->addMonth();
            } else {


                if ($client->expire_date < $current_month_billing_date) {

                    $date = Carbon::parse($current_month_billing_date);
                } else {
                    $date = Carbon::parse($client->expire_date);
                }
            }
        } else {

            $newcheck = BillGenerate::where('client_id', $client->id)
                ->whereBetween('created_at', [today()->firstOfMonth(), today()->endOfMonth()])
                ->whereIn('billing_type', ['new', 'otc'])
                ->first();

            if ($newcheck) {
                $date = Carbon::parse($client->expire_date);
            } else {
                if (Carbon::parse($client->expire_date) < today()->endOfMonth()) {
                    $date = Carbon::parse($client->expire_date)->addMonth();
                }
            }
        }

        // dd($date);
        if (getBillingType() === 'day_to_day') {
            // dd('comes');
            $customerAccount = CustomerAccount::where('client_id', $client->id)->first();
            $client_package = Packages::find($client->package_id);
            $addvanced_balance = abs($customerAccount->dueAmount);

            if (Carbon::parse($client->expire_date) < today() && $addvanced_balance >= ($client_package->package_rate - $client->parmanent_discount)) {
                (new SecondMonthlyBillGenerateForDayToDay)->generate(
                    $client->id,
                    "monthly",
                    $client_package->package_rate,
                    'Monthly Bill Generate ',
                    $client->parmanent_discount,
                    $client_package->client_package,
                    $client_package->package_rate,
                    $client->expire_date
                );


                $customerAccount->dueAmount =  $customerAccount->dueAmount + ($client_package->package_rate - $client->parmanent_discount);
                $customerAccount->save();
                $date = Carbon::parse(today())->addDay($total_day);
            } else {
                $date = Carbon::parse($client->expire_date);
            }
        }

        if (!checkAPI()) {
            (new IdEnableDisableController)->enableInRadius($client);
            DB::table('radcheck')
                ->where('username', '=', $client->userid)
                ->update([
                    'op' => ':='
                ]);
        }

        $client = Client::find($client->id);


        $client->expire_date    = $date;
        $client->billing_cycle = Carbon::parse($date)->format('d');
        // $client->clients_status     = 'active';
        $client->payment_dadeline   = 0;
        $client->save();

        if ($client->expire_date >= today()) {
            $client->clients_status = 'active';
            $client->save();
        }
    }


    public function paymentDelete(Request $request)
    {

        $payment = Billpayment::with('user', 'income')->find($request->payment_id);
        $amount = floatval($payment->paid_amount) + floatval($payment->discount_amount);
        $paid_amount = floatval($payment->paid_amount);

        if ($payment->paid_amount == 0 && $payment->discount_amount == 0) {
            return redirect()->back()->with('success_message', 'Payment and discount both amount are 0 so no need to delete.');
        }

        DB::beginTransaction();

        try {

            $reseller_commission = ResellerCommissionReference::where('bill_payment_id', $payment->id)->first();
            if ($reseller_commission && $reseller_commission->bill_payment_id != null) {
                $reseller_commission->actions = $reseller_commission->actions . " deleted";
                $reseller_commission->received_amount = 0;
                $reseller_commission->save();
            }

            $clientIdForThisBillDelete = PaymentDeleteController::deletePayment($payment, $amount, $paid_amount);

            if (checkSettings('otc_payment_with_generale_payment') == 'enable') {

                $checkOtcPayment = OtcPaymentLog::where('payment_id', $payment->id)->get();
                if (count($checkOtcPayment) > 0) {
                    $clientInfo = Clientsinfo::where('client_id', $payment->client_id)->first();
                    $clientInfo->otc_due = $clientInfo->otc;
                    $clientInfo->save();
                    OtcPaymentLog::where('client_id', $payment->client_id)->delete();
                }
            }


            DB::commit();
            (new BillgenerateUpdate)->update($clientIdForThisBillDelete);

            return redirect()->back()->with('success_message', 'Payment Successfully Reverted.');
        } catch (\Throwable $th) {
            // dd($th);

            DB::rollBack();
            return redirect()->back()->with('success_message', 'Payment Deleted Unsuccessfull.');
        }
    }

    public function dueHistoryInPayment($id)
    {
        $due_months = BillGenerate::where('client_id', $id)
            ->whereColumn('bill_amount', '!=', 'paid_amount')
            ->get();

        $months = [];

        foreach ($due_months as $due_month) {
            $date = Carbon::parse($due_month->created_at)->format('M');
            $amount = $due_month->bill_amount - $due_month->paid_amount;
            // $months[] = "<button class='btn btn-secondary mx-2 btn-sm' disabled type='button'>". $date ."</button>";
            $months[] = "<tr>
                        <td width='70%'>" . $date . "</td>
                        <td width='30%'>" . $amount . "tk</td>
                        </tr>";
        }

        return $months;
    }

    public function invoicePrint($id)
    {
        $billpayment = Billpayment::with('clients', 'clientsinfo', 'employee', 'clients.customerAccount')->find($id);

        $data = [
            'billpayment'  => $billpayment,
            'company_info' => CompanyInformation::latest()->first(),
        ];
        // dd($data);
        return view('billing.printInvoice', $data);
    }

    public function UpaypaymentProcess($amount, $client_id, $trx_id)
    {

        $this->billGenerateIfNotExist($client_id);

        $client = Client::with('clientsinfo', 'pop', 'pop.reseller', 'customerAccount', 'generatedBill')->withCount('generatedBill')->find($client_id);
        $current_due_amount = $client->customerAccount->dueAmount ?? 0;
        // dd($client_id);
        if (!empty($amount)) {

            $data = [
                'description'           =>  'Upay Bill Payment from client portal',
                'paid_amount'           =>  $amount ?? 0,
                'discount_amount'       =>  0,
                'client_id'             =>  $client_id,
                'user_id'               =>  User::onlinePaymentUser('Upay-Online')->id,
                'created_at'            =>  now(),
                'collected_by'          =>  Employee::onlinePaymentEmployee('Upay-Online')->id,
                'money_receipt_number'  =>  $trx_id,
                'expire_date_history'   =>  $client->expire_date,
                'payment_method'        =>  'upay',
                'client_id_time'        =>  $client_id . '_' . now()->format('Y-m-d H:i')
            ];
            $payment = Billpayment::create($data);

            CustomerAccount::updateCustomrAccount($client_id, (-$amount), $amount, 0);

            if ($client->pop->reseller->reseller_type == 'own') {

                $actions = 'Bill Payment. cost ' . $amount . 'TK for the customer ' . $client->userid;

                $userAccountingId = UserAccounting::create([
                    'received_amount'   => $amount,
                    'paid_amount'       => 0,
                    'actions'           => $actions,
                    'commetns'          => '',
                    'user_id'           => User::onlinePaymentUser('Upay-Online')->id,
                    'client_id'         => $client_id,
                    'accounting_type'   => 'Upay Payment',
                    'transaction_type'  => 'Upay Payment',
                ]);
                $payment->user_accountings_id = $userAccountingId;
            }

            $income_type = IncomeHead::find(11);

            $income =   Income::create([
                'name' => $income_type->name ?? 'Upay Payment',
                'date' => now(),
                'amount' => $amount,
                'description' => 'Upay Bill Payment from customer : ' . $client->userid,
                'incomeHead' => 11
            ]);
            $payment->income_id = $income->id;
            $payment->save();


            $cac = CustomerAccount::where('client_id', $client->id)->first();

            $this->performeExpireDateExtend($client, $payment, $cac, $amount, $client->pop, $current_due_amount, 'billpay', 0);

            if (checkSettings('auto-payment-sms-send') == 'enable') {
                try {
                    (new PaymentSms)->billPaymentSms($client, $amount, $trx_id, $payment->id, 'upay');
                } catch (\Exception $e) {
                }
            }

            Session::flash('success_message', 'Upay Payment is Successful');
            // dd("comes");
        }
    }

    public function NagadClientCheckOutPaymentProcess($amount, $client_id, $trx_id, $mobile_that_use, $payment_ref_id, $payment_created_time)
    {
        $this->billGenerateIfNotExist($client_id);

        $client = Client::with('clientsinfo', 'pop', 'pop.reseller', 'customerAccount', 'generatedBill')->withCount('generatedBill')->find($client_id);
        $current_due_amount = $client->customerAccount->dueAmount ?? 0;
        // dd($client_id);
        if (!empty($amount)) {

            $data = [
                'description'           =>  'Nagad Bill Payment from client portal',
                'paid_amount'           =>  $amount ?? 0,
                'discount_amount'       =>  0,
                'client_id'             =>  $client_id,
                'user_id'               =>  User::onlinePaymentUser('Nagad-Online')->id,
                'created_at'            =>  now(),
                'collected_by'          =>  Employee::onlinePaymentEmployee('Nagad-Online')->id,
                'money_receipt_number'  =>  $trx_id,
                'expire_date_history'   =>  $client->expire_date,
                'payment_method'        =>  'nagad',
                'client_id_time'        =>  $client_id . '_' . now()->format('Y-m-d H:i'),
                'trx_id'                =>  $payment_ref_id,
                'customerMsisdn'       =>  $mobile_that_use,
                'paymentCreateTime'    =>  $payment_created_time,
            ];
            $payment = Billpayment::create($data);

            CustomerAccount::updateCustomrAccount($client_id, (-$amount), $amount, 0);

            if ($client->pop->reseller->reseller_type == 'own') {

                $actions = 'Bill Payment. cost ' . $amount . 'TK for the customer ' . $client->userid;

                $userAccountingId = UserAccounting::create([
                    'received_amount'   => $amount,
                    'paid_amount'       => 0,
                    'actions'           => $actions,
                    'commetns'          => '',
                    'user_id'           => User::onlinePaymentUser('Nagad-Online')->id,
                    'client_id'         => $client_id,
                    'accounting_type'   => 'Nagad Payment',
                    'transaction_type'  => 'Nagad Payment',
                ]);
                $payment->user_accountings_id = $userAccountingId;
            }

            $income_type = IncomeHead::find(11);

            $income =   Income::create([
                'name' => $income_type->name ?? 'Nagad Payment',
                'date' => now(),
                'amount' => $amount,
                'description' => 'Nagad Bill Payment from customer : ' . $client->userid,
                'incomeHead' => 11
            ]);
            $payment->income_id = $income->id;
            $payment->save();


            $cac = CustomerAccount::where('client_id', $client->id)->first();

            $this->performeExpireDateExtend($client, $payment, $cac, $amount, $client->pop, $current_due_amount, 'billpay', 0);

            if (checkSettings('auto-payment-sms-send') == 'enable') {
                try {
                    (new PaymentSms)->billPaymentSms($client, $amount, $trx_id, $payment->id, 'nagad');
                } catch (\Exception $e) {
                }
            }

            Session::flash('success_message', 'Nagad Payment is Successful');
            // dd("comes");
        }
    }

    public function surjoPayClientCheckOutPaymentProcess($amount, $client_id, $trx_id, $phone_no, $invoice_no, $date_time)
    {
        $this->billGenerateIfNotExist($client_id);

        $client = Client::with('clientsinfo', 'pop', 'pop.reseller', 'customerAccount', 'generatedBill')->withCount('generatedBill')->find($client_id);
        $current_due_amount = $client->customerAccount->dueAmount ?? 0;
        // dd($amount);

        if (!empty($amount)) {

            $data = [
                'description'           =>  'Surjo Pay Bill Payment from client portal',
                'paid_amount'           =>  $amount ?? 0,
                'discount_amount'       =>  0,
                'client_id'             =>  $client_id,
                'user_id'               =>  User::onlinePaymentUser('Surjopay-Online')->id,
                'created_at'            =>  now(),
                'collected_by'          =>  Employee::onlinePaymentEmployee('Surjopay-Online')->id,
                'money_receipt_number'  =>  $invoice_no,
                'expire_date_history'   =>  $client->expire_date,
                'payment_method'        =>  'Surjopay',
                'client_id_time'        =>  $client_id . '_' . now()->format('Y-m-d H:i'),
                'trx_id'                =>  $trx_id,
                'customerMsisdn'       =>  $phone_no,
                'paymentCreateTime'    =>  $date_time,
            ];
            $payment = Billpayment::create($data);

            CustomerAccount::updateCustomrAccount($client_id, (-$amount), $amount, 0,);

            if ($client->pop->reseller->reseller_type == 'own') {

                $actions = 'Bill Payment. cost ' . $amount . 'TK for the customer ' . $client->userid;

                $userAccountingId = UserAccounting::create([
                    'received_amount'   => $amount,
                    'paid_amount'       => 0,
                    'actions'           => $actions,
                    'commetns'          => '',
                    'user_id'           => User::onlinePaymentUser('Surjopay-Online')->id,
                    'client_id'         => $client_id,
                    'accounting_type'   => 'Surjopay Payment',
                    'transaction_type'  => 'Surjopay Payment',
                ]);
                $payment->user_accountings_id = $userAccountingId;
            }

            $income_type = IncomeHead::find(11);

            $income =   Income::create([
                'name' => $income_type->name ?? 'Surjopay Payment',
                'date' => now(),
                'amount' => $amount,
                'description' => 'Surjopay Bill Payment from customer : ' . $client->userid,
                'incomeHead' => 11
            ]);
            $payment->income_id = $income->id;
            $payment->save();


            $cac = CustomerAccount::where('client_id', $client->id)->first();

            $this->performeExpireDateExtend($client, $payment, $cac, $amount, $client->pop, $current_due_amount, 'billpay', 0);

            if (checkSettings('auto-payment-sms-send') == 'enable') {
                try {
                    (new PaymentSms)->billPaymentSms($client, $amount, $trx_id, $payment->id, 'Surjopay');
                } catch (\Exception $e) {
                }
            }

            Session::flash('success_message', 'Surjopay Payment is Successful');
            // dd("comes");
        }
    }

    public function bkashWebhookPaymentProcess($amount, $client_id, $trx_id, $phone_no, $invoice_no, $date_time)
    {
        $this->billGenerateIfNotExist($client_id);

        $client = Client::with('clientsinfo', 'pop', 'pop.reseller', 'customerAccount', 'generatedBill')->withCount('generatedBill')->find($client_id);
        $current_due_amount = $client->customerAccount->dueAmount ?? 0;
        // dd($amount);

        if (!empty($amount)) {

            $data = [
                'description'           =>  'Bill Payment by Bkash Webhook',
                'paid_amount'           =>  $amount ?? 0,
                'discount_amount'       =>  0,
                'client_id'             =>  $client_id,
                'user_id'               =>  User::onlinePaymentUser('Bkash-Webhook')->id,
                'created_at'            =>  now(),
                'collected_by'          =>  Employee::onlinePaymentEmployee('Bkash-Webhook')->id,
                'money_receipt_number'  =>  $invoice_no,
                'expire_date_history'   =>  $client->expire_date,
                'payment_method'        =>  'Bkash Webhook',
                'client_id_time'        =>  $client_id . '_' . now()->format('Y-m-d H:i'),
                'trx_id'                =>  $trx_id,
                'customerMsisdn'       =>  $phone_no,
                'paymentCreateTime'    =>  $date_time,
            ];
            $payment = Billpayment::create($data);

            CustomerAccount::updateCustomrAccount($client_id, (-$amount), $amount, 0,);

            if ($client->pop->reseller->reseller_type == 'own') {

                $actions = 'Bill Payment. cost ' . $amount . 'TK for the customer ' . $client->userid;

                $userAccountingId = UserAccounting::create([
                    'received_amount'   => $amount,
                    'paid_amount'       => 0,
                    'actions'           => $actions,
                    'commetns'          => '',
                    'user_id'           => User::onlinePaymentUser('Bkash-Webhook')->id,
                    'client_id'         => $client_id,
                    'accounting_type'   => 'Bkash-Webhook',
                    'transaction_type'  => 'Bkash-Webhook',
                ]);
                $payment->user_accountings_id = $userAccountingId;
            }

            $income_type = IncomeHead::find(11);

            $income =   Income::create([
                'name' => $income_type->name ?? 'Bkash-Webhook',
                'date' => now(),
                'amount' => $amount,
                'description' => 'Bkash-Webhook Bill Payment from customer : ' . $client->userid,
                'incomeHead' => 11
            ]);
            $payment->income_id = $income->id;
            $payment->save();


            $cac = CustomerAccount::where('client_id', $client->id)->first();

            $this->performeExpireDateExtend($client, $payment, $cac, $amount, $client->pop, $current_due_amount, 'Webhook', 0);

            if (checkSettings('auto-payment-sms-send') == 'enable') {
                try {
                    (new PaymentSms)->billPaymentSms($client, $amount, $trx_id, $payment->id, 'Bkash-Webhook');
                } catch (\Exception $e) {
                }
            }

            Session::flash('success_message', 'Bkash-Webhook Payment is Successful');
            // dd("comes");
        }
    }
}
