<?php

namespace App\Http\Controllers;

use App\Classes\EditLogHistory;
use App\Classes\SMS\ResellerAccountRechargeSms;
use Exception;
use App\Models\Balance;
use App\Models\Packages;
use App\Models\Pop;
use App\Models\Reseller;
use App\Models\ResellerRechargeReport;
use App\Models\UserAccounting;
use Illuminate\Http\Request;
use Database\Seeders\UserHasRole;
use Illuminate\Support\Facades\DB;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Support\Facades\Artisan;

class ResellerController extends Controller
{


    public function __construct()
    {
        $this->middleware('permission:reseller_index|reseller_create|reseller_edit|reseller_destroy|specific_management_services', ['only' => ['index', 'show']]);
        $this->middleware('permission:reseller_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:reseller_edit|manager-accounts-edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:reseller_destroy', ['only' => ['destroy']]);
        $this->middleware('permission:reseller-recharge', ['only' => ['resellerRecharge', 'resellerRechargeUpdate']]);
        $this->middleware('permission:package-permission|specific_management_services', ['only' => ['permission', 'permissionUpdate']]);
    }


    public function index()
    {
        $resellers = Reseller::resellerList();
        //$resellers = $t->with('')

        $due = ResellerRechargeReport::selectRaw('sum(amount) as tamount,sum(paid_amount) as tpaid,reseller_id')
            ->groupBy('reseller_id')
            ->get();


        return view('reseller.index', [
            'resellers' => $resellers->paginate(5000),
            //'rechargeList' => $rechargeList,
            'page_title' => 'Manager List',
            'paid_amount' => false,
            'due' => $due
        ]);
    }


    public function create()
    {
        return view('reseller.create', [
            'page_title' => 'Add New Reseller',
            'packages'  => Packages::all()
        ]);
    }

    public function store(Request $request)
    {
        $this->validate(
            $request,
            [
                'name' => 'required',
                'reseller_type' => 'required',
                'address' => 'required',
                'contact' => 'required',
                'billable' => 'required'
            ],
        );

        $payment = json_encode([
            'bkash_charges' => $request->bkash_charges,
            'rocket_charges' => $request->rocket_charges,
            'nagad_charges' => $request->nagad_charges,
            'upay_charges' => $request->upay_charges,
            'ucash_charges' => $request->ucash_charges,
        ]);


        Reseller::create([
            'name'          => $request->name,
            'reseller_type' => $request->reseller_type,
            'address'       => $request->address,
            'contact'       => $request->contact,
            'remark'        => $request->remark,
            'billable'      => $request->billable,
            'payment_charges' => $payment
        ]);

        Artisan::call('cache:clear');

        Toastr::success('Reseller Added Successufll', 'Success');
        return redirect()->route('resellers.index');
    }

    public function show(Reseller $reseller)
    {
        //
    }

    public function edit(Reseller $reseller)
    {
        $reseller = Reseller::find($reseller->id);

        $payment = json_decode($reseller->payment_charges);
        return view('reseller.update', [
            'reseller' => $reseller,
            'page_title' => 'Update Manager Information',
            'payment' => $payment
        ]);
    }

    public function update(Request $request, $id)
    {
        $this->validate(
            $request,
            [
                'name' => 'required',
                'reseller_type' => 'required',
                'address' => 'required',
                'contact' => 'required',
                'billable' => 'required'
            ]
        );



        DB::beginTransaction();

        $type = 'Manager';
        $old_info = Reseller::where('id', $id)->first();

        $payment = json_encode([
            'bkash_charges' => $request->bkash_charges,
            'rocket_charges' => $request->rocket_charges,
            'nagad_charges' => $request->nagad_charges,
            'upay_charges' => $request->upay_charges,
            'ucash_charges' => $request->ucash_charges,
        ]);

        $reseller = Reseller::find($id);
        $reseller->update([
            'name'       => $request->name,
            'reseller_type' => $request->reseller_type,
            'address'    => $request->address,
            'contact'    => $request->contact,
            'remark'     => $request->remark,
            'billable'   => $request->billable,
            'payment_charges' => $payment
        ]);

        $new_info = Reseller::find($reseller->id);
        (new EditLogHistory)->editLogSave($reseller, $type, $old_info, $new_info);

        $bill_generate = $request->reseller_type == 'own' ? 'yes' : 'no';
        Pop::where('reseller_id', $id)->update(['bill_generate' => $bill_generate]);
        DB::commit();
        Artisan::call('cache:clear');

        Toastr::success('Reseller Update Successufll', 'Success');
        return redirect()->route('resellers.index');
    }


    public function destroy(Reseller $reseller)
    {
        //
    }

    public function permission($id)
    {
        $allPackage = Packages::all();

        $reseller = Reseller::find($id);

        return view('reseller.package', [
            'reseller' => $reseller,
            'packages' => $allPackage,
            'page_title' => 'Assign Package Permission'
        ]);
    }


    public function permissionUpdate(Request $request)
    {
        // dd($request->all());
        $this->validate($request, [
            'reseller_id' => 'required'
        ]);



        try {

            if (isset($request->package_id) && !empty($request->package_id)) {
                $package_list = implode(',', $request->package_id);
            } else {
                $package_list = '';
            }
            // dd($package_list);
            $reseller = Reseller::find($request->reseller_id);
            $reseller->package_list = $package_list;
            $reseller->save();

            clearCache('reseller');

            return redirect()->route('reseller.permission', $request->reseller_id)->with('success_message', 'Package Assigned Success');

            // }
        } catch (Exception $e) {
            dd($e);
            return redirect()->route('reseller.permission', $request->reseller_id)
                ->with('error_message', 'Some Thing is wrong!!');
        }
    }



    public function resellerRecharge($id, Request $request)
    {
        $reseller = Reseller::find($id);
        $rechargeList = ResellerRechargeReport::with('reseller', 'user', 'balance')
            ->where('reseller_id', $id)
            ->orderBy('id', 'desc')
            ->take(10);

        return view('reseller.resellerRecharge', [
            'page_title'   => 'Manager Recharge',
            'reseller'     => $reseller,
            'rechargeList' =>  $rechargeList->get(),
            'paid_amount' => false
        ]);
    }



    public function resellerRechargeUpdate(Request $request)
    {

        // dd($request->all());
        $txn_id = now()->format('YmdHi');

        $this->validate($request, [
            'amount' => 'required|numeric'
        ]);

        if (ResellerRechargeReport::where('txn_id', $txn_id)->count() > 0 && checkSettings('reseller-recharge-duplicate-protect') == 'enable') {
            Toastr::error('Duplicate Transaction', 'Error');
            return redirect()->back();
        }


        if (checkSettings('reseller-recharge-paid-option') == 'disable') {
            $paid_amount = $request->amount;
        } else {
            $paid_amount = $request->paid_amount;
        }

        $reseller = Reseller::find($request->type_id);


        if ($reseller->reseller_type == 'other') {
            $actions = 'Reseller payment received from ' . $reseller->name . ' ' . $reseller->contact;
            UserAccounting::userAcStore($paid_amount, $actions);
        }





        $status = Balance::balanceUpdate($request->type, $request->type_id, $request->amount, $request->remark, $paid_amount,'',$request->recharge_type,null,null,$txn_id);
        clearCache('reseller');

        $totalRecharge = ResellerRechargeReport::with('reseller', 'user', 'balance')->where('reseller_id', $reseller->id)->sum('amount');

        if ($status === 'success') {
            $sms_setting = getSmsMessage()->where('tamplate_name', 'reseller_balance_recharge_sms')->count() > 0 ? json_decode(getSmsMessage()->where('tamplate_name', 'reseller_balance_recharge_sms')->first()->tamplate_body, true) : '';
            if (!empty($sms_setting) && $sms_setting["sendsms"] == "Yes") {
                try{

                    (new ResellerAccountRechargeSms())->sendSms($reseller->id, $request->amount, $totalRecharge);
                }catch(\Exception $e){

                }
            }

            return redirect()->route('reseller.recharge', $request->type_id)->with('success_message', 'Recharge Success');
        } else {
            return redirect()->route('reseller.recharge', $request->type_id)->with('error_message', $status);
        }
    }

    public function managerBalance()
    {
        // $managerBalance = Reseller::all();
        $managerBalance = DB::table('resellers')
            ->join('reseller_recharge_reports', 'resellers.id', '=', 'reseller_recharge_reports.reseller_id')
            ->select('resellers.*', 'reseller_recharge_reports.after_recharge_balance')
            ->get();
        return view('billing.managerBalance', ['managerBalance' => $managerBalance]);
    }

    public function recharge()
    {
    }

    public function resellerPop(Request $request)
    {
        $popes = Pop::where('reseller_id', $request->reseller)->get();
        $temp_packages = Reseller::where('id', $request->reseller)->first();
        $packages = explode(",", $temp_packages->package_list);
        $packages = DB::table('packages')
            ->whereIn('id', $packages)
            ->get();
        $data = [
            'popes' => $popes,
            'packages' => $packages,

        ];

        return $data;
    }
}
